# HomeHelper - Professional Household Services Platform

A modern, professional Symfony 7.1 application for managing household service bookings. Built with best practices, following senior-level development standards.

## Features

### Current Implementation
- **Public Website**
  - Modern, responsive landing page
  - Service provider listings with search and filters
  - Detailed service provider profiles with reviews
  - Booking system with form validation
  - About/How It Works page
  - Mobile-responsive design

- **Service Provider Dashboard** (`ROLE_SERVICE_PROVIDER`) ✅ **LIVE**
  - Secure login/logout system
  - Dashboard with statistics (total bookings, pending, completed, ratings)
  - **Edit Profile**: Update name, phone, bio, years of experience, and hourly rate
  - **Manage Services**: Add/remove services offered (checkboxes for all available services)
  - **View Bookings**: Complete list of all customer bookings with details
  - Responsive admin interface with sidebar navigation
  - Real-time profile updates
  - View public profile directly from dashboard

- **Super Admin Dashboard** (`ROLE_ADMIN`, `ROLE_SUPER_ADMIN`) ✅ **LIVE**
  - **Dashboard**: Complete statistics overview with earnings tracking
    - Total earnings from completed bookings
    - Pending earnings from confirmed/pending bookings
    - Booking statistics (total, pending, confirmed, completed, cancelled)
    - Service provider statistics (total, active)
  - **Manage Bookings**: Full CRUD operations
    - View all bookings in detailed table
    - Create new bookings manually
    - Edit existing bookings (customer info, service, date/time, status)
    - Delete bookings
    - Update booking status (pending → confirmed → completed)
    - Real-time total price calculation
  - **Manage Service Providers**: Complete provider management
    - View all service providers with details
    - Add new service providers (auto-generates account)
    - Edit provider information (name, contact, experience, rate)
    - Set availability and verification status
    - Delete service providers
    - Default password: `password123` for new providers
  - Quick action cards for common tasks
  - Responsive admin interface
  - Role-based access control

- **Database Architecture**
  - User management with roles (Admin, Service Provider, Customer)
  - Service Provider profiles with experience, rates, and availability
  - Service catalog with descriptions
  - Booking system with status tracking
  - Review and rating system
  - Relational database with proper constraints
  - Authentication and authorization with Symfony Security

## Technology Stack

- **Framework:** Symfony 7.1
- **Database:** SQLite (easily switchable to MySQL/PostgreSQL)
- **Template Engine:** Twig
- **ORM:** Doctrine
- **Forms:** Symfony Forms with validation
- **Security:** Symfony Security Component with password hashing
- **Frontend:** Pure CSS with modern responsive design

## Installation

### Prerequisites
- PHP 8.2 or higher
- Composer
- Symfony CLI (optional, for development server)

### Setup Steps

1. **Install dependencies:**
   ```bash
   cd homewifes
   composer install
   ```

2. **Configure database:**
   The application is pre-configured to use SQLite. The database file is located at `var/data.db`.

   To use MySQL or PostgreSQL, update the `DATABASE_URL` in `.env`:
   ```
   # For MySQL:
   DATABASE_URL="mysql://username:password@127.0.0.1:3306/homehelper?serverVersion=8.0&charset=utf8mb4"

   # For PostgreSQL:
   DATABASE_URL="postgresql://username:password@127.0.0.1:5432/homehelper?serverVersion=16&charset=utf8"
   ```

3. **Create database and run migrations:**
   ```bash
   php bin/console doctrine:database:create
   php bin/console doctrine:migrations:migrate
   ```

4. **Load sample data:**
   ```bash
   php bin/console doctrine:fixtures:load
   ```

   This creates:
   - 6 sample service providers with profiles and reviews
   - 8 different services
   - Sample bookings and reviews
   - Admin user (credentials below)

5. **Start the development server:**
   ```bash
   symfony server:start
   ```

   Or use PHP's built-in server:
   ```bash
   php -S localhost:8000 -t public/
   ```

6. **Visit the application:**
   Open your browser and navigate to: `http://localhost:8000`

## Sample Credentials

### Admin User
- Email: `admin@homehelper.com`
- Password: `admin123`
- Roles: `ROLE_ADMIN`, `ROLE_SUPER_ADMIN`

### Service Providers
All service providers use the same password:
- **Password:** `password123`

Available accounts:
- `maria.johnson@example.com` - 8 years experience, $35/hr
- `sarah.chen@example.com` - 5 years experience, $40/hr
- `linda.rodriguez@example.com` - 10 years experience, $38/hr
- `anna.patel@example.com` - 6 years experience, $35/hr
- `emily.williams@example.com` - 7 years experience, $42/hr
- `jessica.kim@example.com` - 4 years experience, $36/hr

## Using the Service Provider Dashboard

### Accessing the Dashboard

1. **Login:**
   - Click "Provider Login" in the top navigation
   - Or visit: `http://localhost:8000/login`
   - Use any service provider credentials listed above

2. **Dashboard Overview:**
   After login, you'll see:
   - Statistics cards (bookings, ratings, etc.)
   - Profile summary
   - Your offered services
   - Recent bookings table

### Managing Your Profile

1. **Click "Edit Profile"** from the sidebar or dashboard
2. Update any of the following:
   - **First Name & Last Name** - Your public display name
   - **Phone Number** - Contact information (optional)
   - **Bio** - Detailed description shown on your public profile
   - **Years of Experience** - Number of years in the industry
   - **Hourly Rate** - Your rate in USD (e.g., 35.00)
3. **Click "Save Changes"**
4. Changes are immediately reflected on your public profile

### Managing Your Services

1. **Click "Manage Services"** from the sidebar
2. **Check/uncheck services** you offer:
   - 🧹 House Cleaning
   - 🪟 Window Cleaning
   - 👔 Laundry & Ironing
   - 🍳 Meal Preparation
   - 📦 Organization
   - 🛒 Grocery Shopping
   - ✨ Deep Cleaning
   - 🐾 Pet Care
3. **Click "Update Services"**
4. Only checked services will appear on your public profile
5. Customers can only book services you've selected

### Viewing Bookings

1. **Click "My Bookings"** from the sidebar
2. See all bookings with:
   - Customer name and contact info
   - Service date and time
   - Service type and duration
   - Total amount
   - Booking status (pending, confirmed, completed, cancelled)

### Viewing Your Public Profile

1. **Click "View Public Profile"** from the sidebar
2. See exactly what customers see when viewing your profile
3. Quick way to verify your changes are displayed correctly

## Project Structure

```
homewifes/
   config/              # Application configuration
   migrations/          # Database migrations
   public/
      css/            # Compiled CSS assets
      index.php       # Application entry point
   src/
      Controller/     # Controllers for routes
      Entity/         # Doctrine entities (database models)
      Form/           # Symfony form types
      Repository/     # Database repositories
      DataFixtures/   # Sample data fixtures
   templates/          # Twig templates
      base.html.twig  # Base template with navbar/footer
      home/           # Landing page
      service_provider/ # Staff listing and details
      about/          # How it works page
      booking/        # Booking confirmation
   var/                # Cache, logs, and database
   .env                # Environment configuration
```

## Database Schema

### Main Entities

- **User**: Authentication and basic user information
  - Supports multiple roles: ADMIN, SUPER_ADMIN, SERVICE_PROVIDER
  - Password hashing with Symfony's security component

- **ServiceProvider**: Extended profile for service providers
  - One-to-one relationship with User
  - Tracks experience, rates, availability, ratings
  - Automatic rating calculation based on reviews

- **Service**: Catalog of available services
  - House Cleaning, Window Cleaning, Meal Preparation, etc.
  - Supports icons and descriptions

- **ServiceProviderService**: Many-to-many relationship
  - Links service providers to the services they offer

- **Booking**: Customer reservations
  - Status tracking: pending, confirmed, cancelled, completed
  - Automatic price calculation
  - Timestamped for audit trail

- **Review**: Customer feedback and ratings
  - Linked to specific bookings
  - Moderation system (isApproved flag)
  - 1-5 star rating system

## Key Features for Future Development

### Service Provider Portal
Ready entities and structure for:
- Login system (already implemented in User entity)
- Profile management (ServiceProvider entity)
- Service selection (ServiceProviderService entity)
- Booking management (Booking entity with status)
- Availability calendar (availability field in ServiceProvider)

### Admin Portal
Ready entities and structure for:
- Booking management and monitoring
- Review moderation (Review.isApproved)
- Service provider verification (ServiceProvider.isVerified)
- User management
- Analytics and reporting

## Development Commands

### Database
```bash
# Create migration after entity changes
php bin/console make:migration

# Run migrations
php bin/console doctrine:migrations:migrate

# Reset database and reload fixtures
php bin/console doctrine:database:drop --force
php bin/console doctrine:database:create
php bin/console doctrine:migrations:migrate
php bin/console doctrine:fixtures:load
```

### Code Generation
```bash
# Create new entity
php bin/console make:entity

# Create new controller
php bin/console make:controller

# Create new form
php bin/console make:form
```

### Cache
```bash
# Clear cache
php bin/console cache:clear
```

## Security Notes

- All passwords are hashed using Symfony's password hasher
- CSRF protection enabled on forms
- SQL injection protection via Doctrine ORM
- XSS protection via Twig auto-escaping
- Role-based access control ready for implementation

## API Routes

### Public Routes
- `/` - Landing page
- `/services` - Service provider listing
- `/staff/{id}` - Service provider detail and booking form
- `/about` - How it works page
- `/booking/create/{serviceProviderId}` - Process booking (POST)
- `/booking/success/{id}` - Booking confirmation

### Future Admin Routes (To Be Implemented)
- `/admin/dashboard` - Admin overview
- `/admin/bookings` - Manage all bookings
- `/admin/providers` - Manage service providers
- `/admin/reviews` - Moderate reviews
- `/provider/dashboard` - Service provider dashboard
- `/provider/bookings` - Provider's bookings
- `/provider/profile` - Manage profile

## Contributing

When adding new features:
1. Create entities first with proper validation
2. Generate migrations and test them
3. Create repositories with custom query methods
4. Build controllers with proper error handling
5. Create responsive Twig templates
6. Add fixtures for testing
7. Document any new routes or features

## License

Proprietary - All rights reserved

## Support

For issues or questions, please contact the development team.

---

**Built with Symfony 7.1** | Following industry best practices and senior-level development standards
