<?php

namespace App\Controller;

use App\Entity\Booking;
use App\Form\BookingType;
use App\Repository\BookingRepository;
use App\Repository\ServiceProviderRepository;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;

class BookingController extends AbstractController
{
    #[Route('/booking/create/{serviceProviderId}', name: 'app_booking_create', methods: ['POST'])]
    public function create(
        int $serviceProviderId,
        Request $request,
        ServiceProviderRepository $serviceProviderRepository,
        EntityManagerInterface $entityManager
    ): Response {
        $serviceProvider = $serviceProviderRepository->find($serviceProviderId);

        if (!$serviceProvider) {
            throw $this->createNotFoundException('Service provider not found');
        }

        $booking = new Booking();
        $booking->setServiceProvider($serviceProvider);

        $form = $this->createForm(BookingType::class, $booking);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            // Calculate total price
            $booking->calculateTotalPrice();
            $booking->setStatus(Booking::STATUS_PENDING);

            $entityManager->persist($booking);
            $entityManager->flush();

            $this->addFlash('success', 'Your booking request has been submitted successfully! We will confirm your appointment within 24 hours.');

            return $this->redirectToRoute('app_booking_success', ['id' => $booking->getId()]);
        }

        // If form is not valid, redirect back to staff detail page
        $this->addFlash('error', 'There was an error with your booking. Please check all fields and try again.');

        return $this->redirectToRoute('app_staff_detail', ['id' => $serviceProviderId]);
    }

    #[Route('/booking/success/{id}', name: 'app_booking_success')]
    public function success(int $id, EntityManagerInterface $entityManager): Response
    {
        $booking = $entityManager->getRepository(Booking::class)->find($id);

        if (!$booking) {
            throw $this->createNotFoundException('Booking not found');
        }

        return $this->render('booking/success.html.twig', [
            'booking' => $booking,
        ]);
    }

    #[Route('/api/bookings/provider/{serviceProviderId}', name: 'app_api_bookings_provider', methods: ['GET'])]
    public function getProviderBookings(
        int $serviceProviderId,
        BookingRepository $bookingRepository
    ): JsonResponse {
        // Get all confirmed and pending bookings for this provider
        $bookings = $bookingRepository->createQueryBuilder('b')
            ->where('b.serviceProvider = :providerId')
            ->andWhere('b.status IN (:statuses)')
            ->setParameter('providerId', $serviceProviderId)
            ->setParameter('statuses', [Booking::STATUS_PENDING, Booking::STATUS_CONFIRMED])
            ->orderBy('b.serviceDate', 'ASC')
            ->getQuery()
            ->getResult();

        $events = [];
        foreach ($bookings as $booking) {
            $startDateTime = \DateTime::createFromInterface($booking->getServiceDate());
            $time = \DateTime::createFromInterface($booking->getServiceTime());
            $startDateTime->setTime((int)$time->format('H'), (int)$time->format('i'));

            $endDateTime = clone $startDateTime;
            $endDateTime->modify('+' . $booking->getHours() . ' hours');

            $events[] = [
                'id' => $booking->getId(),
                'title' => $booking->getService()->getName(),
                'start' => $startDateTime->format('Y-m-d\TH:i:s'),
                'end' => $endDateTime->format('Y-m-d\TH:i:s'),
                'backgroundColor' => $booking->isConfirmed() ? '#667eea' : '#f59e0b',
                'borderColor' => $booking->isConfirmed() ? '#667eea' : '#f59e0b',
                'status' => $booking->getStatus(),
            ];
        }

        return new JsonResponse($events);
    }
}
