<?php

namespace App\Controller;

use App\Entity\User;
use App\Entity\ServiceProvider;
use App\Form\RegistrationFormType;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\PasswordHasher\Hasher\UserPasswordHasherInterface;
use Symfony\Component\Routing\Attribute\Route;
use Doctrine\DBAL\Exception\UniqueConstraintViolationException;

class RegistrationController extends AbstractController
{
    public function __construct(
        private UserPasswordHasherInterface $passwordHasher,
        private EntityManagerInterface $entityManager
    ) {
    }

    #[Route('/register', name: 'app_register')]
    public function register(Request $request): Response
    {
        // Redirect if already logged in
        if ($this->getUser()) {
            return $this->redirectToRoute('app_provider_dashboard');
        }

        $user = new User();
        $form = $this->createForm(RegistrationFormType::class, $user);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            try {
                // Hash the password
                $plainPassword = $form->get('plainPassword')->getData();
                $hashedPassword = $this->passwordHasher->hashPassword(
                    $user,
                    $plainPassword
                );
                $user->setPassword($hashedPassword);

                // Set role for service provider
                $user->setRoles(['ROLE_SERVICE_PROVIDER']);

                // Set user as active by default
                $user->setIsActive(true);

                // Create ServiceProvider entity
                $serviceProvider = new ServiceProvider();
                $serviceProvider->setUser($user);

                // Set optional service provider fields if provided
                $bio = $form->get('bio')->getData();
                if ($bio) {
                    $serviceProvider->setBio($bio);
                }

                $yearsOfExperience = $form->get('yearsOfExperience')->getData();
                if ($yearsOfExperience !== null) {
                    $serviceProvider->setYearsOfExperience((int) $yearsOfExperience);
                }

                $hourlyRate = $form->get('hourlyRate')->getData();
                if ($hourlyRate !== null) {
                    $serviceProvider->setHourlyRate((float) $hourlyRate);
                }

                // Set default values for service provider
                $serviceProvider->setIsAvailable(true);
                $serviceProvider->setIsVerified(false); // Requires admin verification
                $serviceProvider->setRating(0);
                $serviceProvider->setTotalReviews(0);

                // Persist both entities
                $this->entityManager->persist($user);
                $this->entityManager->persist($serviceProvider);
                $this->entityManager->flush();

                $this->addFlash(
                    'success',
                    'Registration successful! Your account is pending verification. Please log in to access your dashboard.'
                );

                return $this->redirectToRoute('app_login');

            } catch (UniqueConstraintViolationException $e) {
                $this->addFlash(
                    'error',
                    'This email address is already registered. Please use a different email or try logging in.'
                );
            } catch (\Exception $e) {
                $this->addFlash(
                    'error',
                    'An error occurred during registration. Please try again.'
                );

                // Log the error for debugging (in production)
                // $this->logger->error('Registration error: ' . $e->getMessage());
            }
        }

        return $this->render('registration/register.html.twig', [
            'registrationForm' => $form->createView(),
        ]);
    }
}
