<?php

namespace App\Controller;

use App\Entity\ServiceProviderService;
use App\Form\ManageServicesType;
use App\Form\ServiceProviderProfileType;
use App\Repository\BookingRepository;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\File\Exception\FileException;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Security\Http\Attribute\IsGranted;
use Symfony\Component\String\Slugger\SluggerInterface;

#[Route('/provider')]
#[IsGranted('ROLE_SERVICE_PROVIDER')]
class ServiceProviderAdminController extends AbstractController
{
    #[Route('/dashboard', name: 'app_provider_dashboard')]
    public function dashboard(BookingRepository $bookingRepository): Response
    {
        $user = $this->getUser();
        $serviceProvider = $user->getServiceProvider();

        if (!$serviceProvider) {
            throw $this->createNotFoundException('Service provider profile not found');
        }

        // Get bookings for this service provider
        $bookings = $bookingRepository->findByServiceProvider($serviceProvider->getId());

        // Calculate statistics
        $stats = [
            'total_bookings' => count($bookings),
            'pending_bookings' => count(array_filter($bookings, fn($b) => $b->isPending())),
            'completed_bookings' => count(array_filter($bookings, fn($b) => $b->isCompleted())),
            'total_reviews' => $serviceProvider->getTotalReviews(),
            'average_rating' => $serviceProvider->getRating() ?? 'N/A',
        ];

        return $this->render('provider/dashboard.html.twig', [
            'serviceProvider' => $serviceProvider,
            'bookings' => array_slice($bookings, 0, 10), // Latest 10 bookings
            'stats' => $stats,
        ]);
    }

    #[Route('/profile/edit', name: 'app_provider_profile_edit')]
    public function editProfile(Request $request, EntityManagerInterface $entityManager): Response
    {
        $user = $this->getUser();
        $serviceProvider = $user->getServiceProvider();

        if (!$serviceProvider) {
            throw $this->createNotFoundException('Service provider profile not found');
        }

        $form = $this->createForm(ServiceProviderProfileType::class, $serviceProvider);

        // Pre-populate user fields
        $form->get('firstName')->setData($user->getFirstName());
        $form->get('lastName')->setData($user->getLastName());
        $form->get('phone')->setData($user->getPhone());

        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            // Update user fields
            $user->setFirstName($form->get('firstName')->getData());
            $user->setLastName($form->get('lastName')->getData());
            $user->setPhone($form->get('phone')->getData());

            // Update initials
            $serviceProvider->setInitials(
                strtoupper(substr($user->getFirstName(), 0, 1) . substr($user->getLastName(), 0, 1))
            );

            $serviceProvider->setUpdatedAt(new \DateTimeImmutable());

            $entityManager->flush();

            $this->addFlash('success', 'Your profile has been updated successfully!');

            return $this->redirectToRoute('app_provider_dashboard');
        }

        return $this->render('provider/edit_profile.html.twig', [
            'form' => $form,
            'serviceProvider' => $serviceProvider,
        ]);
    }

    #[Route('/profile/upload-image', name: 'app_provider_upload_image', methods: ['POST'])]
    public function uploadProfileImage(Request $request, EntityManagerInterface $entityManager, SluggerInterface $slugger): JsonResponse
    {
        $user = $this->getUser();
        $serviceProvider = $user->getServiceProvider();

        if (!$serviceProvider) {
            return new JsonResponse(['error' => 'Service provider profile not found'], 404);
        }

        $imageFile = $request->files->get('file');

        if (!$imageFile) {
            return new JsonResponse(['error' => 'No file uploaded'], 400);
        }

        // Check if file is valid
        if (!$imageFile->isValid()) {
            return new JsonResponse(['error' => 'File upload failed: ' . $imageFile->getErrorMessage()], 400);
        }

        // Validate file type
        $allowedMimeTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
        $mimeType = $imageFile->getMimeType();

        if (!$mimeType || !in_array($mimeType, $allowedMimeTypes)) {
            return new JsonResponse(['error' => 'Invalid file type. Only JPEG, PNG, GIF, and WebP images are allowed'], 400);
        }

        // Validate file size (max 5MB)
        if ($imageFile->getSize() > 5 * 1024 * 1024) {
            return new JsonResponse(['error' => 'File size exceeds 5MB'], 400);
        }

        // Delete old profile image if exists
        if ($serviceProvider->getProfileImage()) {
            $oldImagePath = $this->getParameter('kernel.project_dir') . '/public/uploads/profiles/' . $serviceProvider->getProfileImage();
            if (file_exists($oldImagePath)) {
                unlink($oldImagePath);
            }
        }

        // Generate unique filename
        $originalFilename = pathinfo($imageFile->getClientOriginalName(), PATHINFO_FILENAME);
        $safeFilename = $slugger->slug($originalFilename);
        $newFilename = $safeFilename . '-' . uniqid() . '.' . $imageFile->guessExtension();

        // Move the file to the uploads directory
        try {
            $imageFile->move(
                $this->getParameter('kernel.project_dir') . '/public/uploads/profiles',
                $newFilename
            );
        } catch (FileException $e) {
            return new JsonResponse(['error' => 'Failed to upload file'], 500);
        }

        // Update service provider with new image filename
        $serviceProvider->setProfileImage($newFilename);
        $serviceProvider->setUpdatedAt(new \DateTimeImmutable());
        $entityManager->flush();

        return new JsonResponse([
            'success' => true,
            'filename' => $newFilename,
            'url' => '/uploads/profiles/' . $newFilename
        ]);
    }

    #[Route('/services/manage', name: 'app_provider_services_manage')]
    public function manageServices(Request $request, EntityManagerInterface $entityManager): Response
    {
        $user = $this->getUser();
        $serviceProvider = $user->getServiceProvider();

        if (!$serviceProvider) {
            throw $this->createNotFoundException('Service provider profile not found');
        }

        // Get currently selected services
        $currentServices = [];
        foreach ($serviceProvider->getServiceProviderServices() as $sps) {
            $currentServices[] = $sps->getService();
        }

        $form = $this->createForm(ManageServicesType::class);
        $form->get('services')->setData($currentServices);

        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $selectedServices = $form->get('services')->getData();

            // Remove all current service associations
            foreach ($serviceProvider->getServiceProviderServices() as $sps) {
                $entityManager->remove($sps);
            }
            $entityManager->flush();

            // Add new service associations
            foreach ($selectedServices as $service) {
                $sps = new ServiceProviderService();
                $sps->setServiceProvider($serviceProvider);
                $sps->setService($service);
                $entityManager->persist($sps);
            }

            $serviceProvider->setUpdatedAt(new \DateTimeImmutable());
            $entityManager->flush();

            $this->addFlash('success', 'Your services have been updated successfully!');

            return $this->redirectToRoute('app_provider_dashboard');
        }

        return $this->render('provider/manage_services.html.twig', [
            'form' => $form,
            'serviceProvider' => $serviceProvider,
            'currentServices' => $currentServices,
        ]);
    }

    #[Route('/bookings', name: 'app_provider_bookings')]
    public function bookings(BookingRepository $bookingRepository): Response
    {
        $user = $this->getUser();
        $serviceProvider = $user->getServiceProvider();

        if (!$serviceProvider) {
            throw $this->createNotFoundException('Service provider profile not found');
        }

        $bookings = $bookingRepository->findByServiceProvider($serviceProvider->getId());

        return $this->render('provider/bookings.html.twig', [
            'serviceProvider' => $serviceProvider,
            'bookings' => $bookings,
        ]);
    }
}
