<?php

namespace App\Controller;

use App\Entity\Booking;
use App\Entity\ServiceProvider;
use App\Entity\User;
use App\Form\AdminBookingType;
use App\Form\AdminServiceProviderType;
use App\Repository\BookingRepository;
use App\Repository\ServiceProviderRepository;
use App\Repository\UserRepository;
use Doctrine\ORM\EntityManagerInterface;
use Knp\Component\Pager\PaginatorInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Security\Http\Attribute\IsGranted;

#[Route('/admin')]
#[IsGranted('ROLE_ADMIN')]
class SuperAdminController extends AbstractController
{
    #[Route('/dashboard', name: 'app_admin_dashboard')]
    public function dashboard(
        BookingRepository $bookingRepository,
        ServiceProviderRepository $serviceProviderRepository,
        UserRepository $userRepository
    ): Response {
        $bookings = $bookingRepository->findAllWithDetails();
        $serviceProviders = $serviceProviderRepository->findAll();
        $users = $userRepository->findAll();

        // Calculate statistics
        $totalEarnings = array_reduce($bookings, function($carry, $booking) {
            if ($booking->isCompleted()) {
                return $carry + (float)$booking->getTotalPrice();
            }
            return $carry;
        }, 0);

        $pendingEarnings = array_reduce($bookings, function($carry, $booking) {
            if ($booking->isPending() || $booking->isConfirmed()) {
                return $carry + (float)$booking->getTotalPrice();
            }
            return $carry;
        }, 0);

        $stats = [
            'total_bookings' => count($bookings),
            'pending_bookings' => count(array_filter($bookings, fn($b) => $b->isPending())),
            'confirmed_bookings' => count(array_filter($bookings, fn($b) => $b->isConfirmed())),
            'completed_bookings' => count(array_filter($bookings, fn($b) => $b->isCompleted())),
            'cancelled_bookings' => count(array_filter($bookings, fn($b) => $b->isCancelled())),
            'total_service_providers' => count($serviceProviders),
            'active_service_providers' => count(array_filter($serviceProviders, fn($sp) => $sp->isAvailable())),
            'total_users' => count($users),
            'total_earnings' => number_format($totalEarnings, 2),
            'pending_earnings' => number_format($pendingEarnings, 2),
        ];

        // Recent bookings
        $recentBookings = array_slice($bookings, 0, 10);

        return $this->render('admin/dashboard.html.twig', [
            'stats' => $stats,
            'recentBookings' => $recentBookings,
        ]);
    }

    // ==================== SERVICE PROVIDERS MANAGEMENT ====================

    #[Route('/service-providers', name: 'app_admin_service_providers')]
    public function serviceProviders(
        Request $request,
        ServiceProviderRepository $serviceProviderRepository,
        PaginatorInterface $paginator
    ): Response {
        $query = $serviceProviderRepository->createQueryBuilder('sp')
            ->orderBy('sp.id', 'DESC')
            ->getQuery();

        $pagination = $paginator->paginate(
            $query,
            $request->query->getInt('page', 1),
            10
        );

        return $this->render('admin/service_providers/list.html.twig', [
            'serviceProviders' => $pagination,
        ]);
    }

    #[Route('/service-providers/new', name: 'app_admin_service_provider_new')]
    public function newServiceProvider(Request $request, EntityManagerInterface $entityManager): Response
    {
        $user = new User();
        $serviceProvider = new ServiceProvider();

        $form = $this->createForm(AdminServiceProviderType::class, $serviceProvider);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            // Create user from form data
            $user->setEmail($form->get('email')->getData());
            $user->setFirstName($form->get('firstName')->getData());
            $user->setLastName($form->get('lastName')->getData());
            $user->setPhone($form->get('phone')->getData());
            $user->setRoles(['ROLE_SERVICE_PROVIDER']);

            // Set default password
            $user->setPassword(
                password_hash('password123', PASSWORD_BCRYPT)
            );

            $entityManager->persist($user);

            // Link service provider to user
            $serviceProvider->setUser($user);
            $serviceProvider->setInitials(
                strtoupper(substr($user->getFirstName(), 0, 1) . substr($user->getLastName(), 0, 1))
            );

            $entityManager->persist($serviceProvider);
            $entityManager->flush();

            $this->addFlash('success', 'Service provider created successfully! Default password is: password123');

            return $this->redirectToRoute('app_admin_service_providers');
        }

        return $this->render('admin/service_providers/form.html.twig', [
            'form' => $form,
            'serviceProvider' => null,
        ]);
    }

    #[Route('/service-providers/{id}/edit', name: 'app_admin_service_provider_edit')]
    public function editServiceProvider(
        ServiceProvider $serviceProvider,
        Request $request,
        EntityManagerInterface $entityManager
    ): Response {
        $form = $this->createForm(AdminServiceProviderType::class, $serviceProvider);

        // Pre-populate user fields
        $form->get('email')->setData($serviceProvider->getUser()->getEmail());
        $form->get('firstName')->setData($serviceProvider->getUser()->getFirstName());
        $form->get('lastName')->setData($serviceProvider->getUser()->getLastName());
        $form->get('phone')->setData($serviceProvider->getUser()->getPhone());

        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $user = $serviceProvider->getUser();
            $user->setEmail($form->get('email')->getData());
            $user->setFirstName($form->get('firstName')->getData());
            $user->setLastName($form->get('lastName')->getData());
            $user->setPhone($form->get('phone')->getData());

            $serviceProvider->setInitials(
                strtoupper(substr($user->getFirstName(), 0, 1) . substr($user->getLastName(), 0, 1))
            );
            $serviceProvider->setUpdatedAt(new \DateTimeImmutable());

            $entityManager->flush();

            $this->addFlash('success', 'Service provider updated successfully!');

            return $this->redirectToRoute('app_admin_service_providers');
        }

        return $this->render('admin/service_providers/form.html.twig', [
            'form' => $form,
            'serviceProvider' => $serviceProvider,
        ]);
    }

    #[Route('/service-providers/{id}/delete', name: 'app_admin_service_provider_delete', methods: ['POST'])]
    public function deleteServiceProvider(
        ServiceProvider $serviceProvider,
        Request $request,
        EntityManagerInterface $entityManager
    ): Response {
        if ($this->isCsrfTokenValid('delete' . $serviceProvider->getId(), $request->request->get('_token'))) {
            $user = $serviceProvider->getUser();

            $entityManager->remove($serviceProvider);
            $entityManager->remove($user);
            $entityManager->flush();

            $this->addFlash('success', 'Service provider deleted successfully!');
        }

        return $this->redirectToRoute('app_admin_service_providers');
    }

    // ==================== BOOKINGS MANAGEMENT ====================

    #[Route('/bookings', name: 'app_admin_bookings')]
    public function bookings(
        Request $request,
        BookingRepository $bookingRepository,
        PaginatorInterface $paginator
    ): Response {
        $query = $bookingRepository->createQueryBuilder('b')
            ->leftJoin('b.serviceProvider', 'sp')
            ->leftJoin('sp.user', 'u')
            ->leftJoin('b.service', 's')
            ->orderBy('b.id', 'DESC')
            ->getQuery();

        $pagination = $paginator->paginate(
            $query,
            $request->query->getInt('page', 1),
            10
        );

        return $this->render('admin/bookings/list.html.twig', [
            'bookings' => $pagination,
        ]);
    }

    #[Route('/bookings/new', name: 'app_admin_booking_new')]
    public function newBooking(Request $request, EntityManagerInterface $entityManager): Response
    {
        $booking = new Booking();

        $form = $this->createForm(AdminBookingType::class, $booking);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $booking->calculateTotalPrice();

            $entityManager->persist($booking);
            $entityManager->flush();

            $this->addFlash('success', 'Booking created successfully!');

            return $this->redirectToRoute('app_admin_bookings');
        }

        return $this->render('admin/bookings/form.html.twig', [
            'form' => $form,
            'booking' => null,
        ]);
    }

    #[Route('/bookings/{id}/edit', name: 'app_admin_booking_edit')]
    public function editBooking(
        Booking $booking,
        Request $request,
        EntityManagerInterface $entityManager
    ): Response {
        $form = $this->createForm(AdminBookingType::class, $booking);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $booking->calculateTotalPrice();
            $booking->setUpdatedAt(new \DateTimeImmutable());

            // Update status timestamps
            if ($booking->isConfirmed() && !$booking->getConfirmedAt()) {
                $booking->setConfirmedAt(new \DateTimeImmutable());
            }
            if ($booking->isCompleted() && !$booking->getCompletedAt()) {
                $booking->setCompletedAt(new \DateTimeImmutable());
            }

            $entityManager->flush();

            $this->addFlash('success', 'Booking updated successfully!');

            return $this->redirectToRoute('app_admin_bookings');
        }

        return $this->render('admin/bookings/form.html.twig', [
            'form' => $form,
            'booking' => $booking,
        ]);
    }

    #[Route('/bookings/{id}/delete', name: 'app_admin_booking_delete', methods: ['POST'])]
    public function deleteBooking(
        Booking $booking,
        Request $request,
        EntityManagerInterface $entityManager
    ): Response {
        if ($this->isCsrfTokenValid('delete' . $booking->getId(), $request->request->get('_token'))) {
            $entityManager->remove($booking);
            $entityManager->flush();

            $this->addFlash('success', 'Booking deleted successfully!');
        }

        return $this->redirectToRoute('app_admin_bookings');
    }
}
