<?php

namespace App\DataFixtures;

use App\Entity\Booking;
use App\Entity\Review;
use App\Entity\Service;
use App\Entity\ServiceProvider;
use App\Entity\ServiceProviderService;
use App\Entity\User;
use Doctrine\Bundle\FixturesBundle\Fixture;
use Doctrine\Persistence\ObjectManager;
use Symfony\Component\PasswordHasher\Hasher\UserPasswordHasherInterface;

class AppFixtures extends Fixture
{
    public function __construct(
        private UserPasswordHasherInterface $passwordHasher
    ) {
    }

    public function load(ObjectManager $manager): void
    {
        // Create Services
        $services = $this->createServices($manager);

        // Create Service Providers
        $serviceProviders = $this->createServiceProviders($manager, $services);

        // Create Sample Bookings and Reviews
        $this->createBookingsAndReviews($manager, $serviceProviders, $services);

        // Create Admin User
        $this->createAdminUser($manager);

        $manager->flush();
    }

    private function createServices(ObjectManager $manager): array
    {
        $servicesData = [
            ['name' => 'House Cleaning', 'description' => 'Comprehensive cleaning including dusting, vacuuming, mopping, and sanitizing all rooms.', 'icon' => '🧹'],
            ['name' => 'Window Cleaning', 'description' => 'Interior and exterior window cleaning for sparkling, streak-free results.', 'icon' => '🪟'],
            ['name' => 'Laundry & Ironing', 'description' => 'Washing, drying, folding, and ironing clothes with care and attention.', 'icon' => '👔'],
            ['name' => 'Meal Preparation', 'description' => 'Planning and preparing delicious, nutritious meals according to your preferences.', 'icon' => '🍳'],
            ['name' => 'Organization', 'description' => 'Organizing closets, pantries, and living spaces for maximum efficiency.', 'icon' => '📦'],
            ['name' => 'Grocery Shopping', 'description' => 'Shopping for groceries based on your list and dietary preferences.', 'icon' => '🛒'],
            ['name' => 'Deep Cleaning', 'description' => 'Thorough, detailed cleaning of your entire home including hard-to-reach areas.', 'icon' => '✨'],
            ['name' => 'Pet Care', 'description' => 'Feeding, walking, and caring for your pets while you\'re away.', 'icon' => '🐾'],
        ];

        $services = [];
        foreach ($servicesData as $serviceData) {
            $service = new Service();
            $service->setName($serviceData['name']);
            $service->setDescription($serviceData['description']);
            $service->setIcon($serviceData['icon']);
            $service->setIsActive(true);

            $manager->persist($service);
            $services[] = $service;
        }

        return $services;
    }

    private function createServiceProviders(ObjectManager $manager, array $services): array
    {
        $providersData = [
            [
                'firstName' => 'Maria',
                'lastName' => 'Johnson',
                'email' => 'maria.johnson@example.com',
                'phone' => '(555) 123-4567',
                'yearsOfExperience' => 8,
                'hourlyRate' => '35.00',
                'bio' => 'Maria is a dedicated household professional with 8 years of experience providing exceptional cleaning and organization services. She takes pride in her attention to detail and consistently receives outstanding feedback from clients. Maria is reliable, trustworthy, and approaches every job with professionalism and care.',
                'services' => [0, 1, 2, 4],
            ],
            [
                'firstName' => 'Sarah',
                'lastName' => 'Chen',
                'email' => 'sarah.chen@example.com',
                'phone' => '(555) 234-5678',
                'yearsOfExperience' => 5,
                'hourlyRate' => '40.00',
                'bio' => 'Sarah combines her culinary expertise with organizational skills to provide top-notch household services. With 5 years of experience, she specializes in meal preparation and creating efficient, organized living spaces.',
                'services' => [3, 4, 5],
            ],
            [
                'firstName' => 'Linda',
                'lastName' => 'Rodriguez',
                'email' => 'linda.rodriguez@example.com',
                'phone' => '(555) 345-6789',
                'yearsOfExperience' => 10,
                'hourlyRate' => '38.00',
                'bio' => 'With a decade of experience, Linda is known for her meticulous deep cleaning and organizational expertise. She has worked with numerous families and businesses, consistently delivering exceptional results.',
                'services' => [6, 2, 4],
            ],
            [
                'firstName' => 'Anna',
                'lastName' => 'Patel',
                'email' => 'anna.patel@example.com',
                'phone' => '(555) 456-7890',
                'yearsOfExperience' => 6,
                'hourlyRate' => '35.00',
                'bio' => 'Anna brings energy and enthusiasm to every job. Her attention to detail and friendly demeanor make her a favorite among clients. She specializes in regular house cleaning and organization.',
                'services' => [0, 1, 4],
            ],
            [
                'firstName' => 'Emily',
                'lastName' => 'Williams',
                'email' => 'emily.williams@example.com',
                'phone' => '(555) 567-8901',
                'yearsOfExperience' => 7,
                'hourlyRate' => '42.00',
                'bio' => 'Emily is a versatile professional who excels in both cooking and cleaning. Her clients appreciate her flexibility and ability to handle multiple household tasks efficiently.',
                'services' => [3, 0, 5],
            ],
            [
                'firstName' => 'Jessica',
                'lastName' => 'Kim',
                'email' => 'jessica.kim@example.com',
                'phone' => '(555) 678-9012',
                'yearsOfExperience' => 4,
                'hourlyRate' => '36.00',
                'bio' => 'Jessica is passionate about creating clean, organized spaces and caring for pets. Her gentle approach with animals and thorough cleaning make her an excellent choice for pet owners.',
                'services' => [6, 2, 7],
            ],
        ];

        $serviceProviders = [];
        foreach ($providersData as $providerData) {
            $user = new User();
            $user->setEmail($providerData['email']);
            $user->setFirstName($providerData['firstName']);
            $user->setLastName($providerData['lastName']);
            $user->setPhone($providerData['phone']);
            $user->setRoles(['ROLE_SERVICE_PROVIDER']);
            $user->setPassword(
                $this->passwordHasher->hashPassword($user, 'password123')
            );
            $manager->persist($user);

            $serviceProvider = new ServiceProvider();
            $serviceProvider->setUser($user);
            $serviceProvider->setBio($providerData['bio']);
            $serviceProvider->setYearsOfExperience($providerData['yearsOfExperience']);
            $serviceProvider->setHourlyRate($providerData['hourlyRate']);
            $serviceProvider->setInitials(
                strtoupper(substr($providerData['firstName'], 0, 1) . substr($providerData['lastName'], 0, 1))
            );
            $serviceProvider->setIsAvailable(true);
            $serviceProvider->setIsVerified(true);
            $manager->persist($serviceProvider);

            foreach ($providerData['services'] as $serviceIndex) {
                $sps = new ServiceProviderService();
                $sps->setServiceProvider($serviceProvider);
                $sps->setService($services[$serviceIndex]);
                $manager->persist($sps);
            }

            $serviceProviders[] = $serviceProvider;
        }

        return $serviceProviders;
    }

    private function createBookingsAndReviews(ObjectManager $manager, array $serviceProviders, array $services): void
    {
        $reviewsData = [
            [
                'serviceProvider' => 0,
                'customerName' => 'Jennifer M.',
                'rating' => 5,
                'comment' => 'Maria is absolutely wonderful! My house has never been cleaner. She\'s reliable, professional, and pays attention to every detail.',
            ],
            [
                'serviceProvider' => 0,
                'customerName' => 'David K.',
                'rating' => 5,
                'comment' => 'Highly recommend! Maria is thorough and efficient. She completed a deep clean of our home in record time.',
            ],
            [
                'serviceProvider' => 1,
                'customerName' => 'Michael R.',
                'rating' => 5,
                'comment' => 'Sarah\'s meal prep service has been a game changer for our family. Delicious, healthy meals every week!',
            ],
            [
                'serviceProvider' => 1,
                'customerName' => 'Lisa T.',
                'rating' => 4,
                'comment' => 'Very organized and professional. Sarah helped us reorganize our entire pantry and kitchen.',
            ],
            [
                'serviceProvider' => 2,
                'customerName' => 'Robert S.',
                'rating' => 5,
                'comment' => 'Linda did an amazing deep clean of our home before we moved in. Everything was spotless!',
            ],
            [
                'serviceProvider' => 3,
                'customerName' => 'Susan B.',
                'rating' => 5,
                'comment' => 'Anna is fantastic! She\'s always on time, friendly, and does an excellent job cleaning our windows.',
            ],
            [
                'serviceProvider' => 4,
                'customerName' => 'James L.',
                'rating' => 5,
                'comment' => 'Emily\'s cooking is outstanding. She accommodates all our dietary restrictions and the food is delicious.',
            ],
            [
                'serviceProvider' => 5,
                'customerName' => 'Patricia W.',
                'rating' => 5,
                'comment' => 'Jessica is wonderful with our pets and keeps our house spotless. We couldn\'t ask for better service!',
            ],
        ];

        foreach ($reviewsData as $index => $reviewData) {
            $sp = $serviceProviders[$reviewData['serviceProvider']];

            $booking = new Booking();
            $booking->setServiceProvider($sp);
            $booking->setService($services[0]);
            $booking->setCustomerName($reviewData['customerName']);
            $booking->setCustomerEmail(strtolower(str_replace([' ', '.'], ['', ''], $reviewData['customerName'])) . '@example.com');
            $booking->setCustomerPhone('(555) 999-' . str_pad($index, 4, '0', STR_PAD_LEFT));

            $date = new \DateTime('-' . rand(10, 60) . ' days');
            $booking->setServiceDate($date);
            $booking->setServiceTime(new \DateTime('09:00'));
            $booking->setHours(4);
            $booking->setServiceAddress('123 Main Street, Apt ' . ($index + 1) . ', City, State 12345');
            $booking->setStatus(Booking::STATUS_COMPLETED);
            $booking->setCompletedAt(new \DateTimeImmutable('-' . rand(5, 55) . ' days'));
            $booking->calculateTotalPrice();

            $manager->persist($booking);

            $review = new Review();
            $review->setServiceProvider($sp);
            $review->setBooking($booking);
            $review->setCustomerName($reviewData['customerName']);
            $review->setRating($reviewData['rating']);
            $review->setComment($reviewData['comment']);
            $review->setIsApproved(true);

            $manager->persist($review);

            $sp->addReview($review);
        }

        foreach ($serviceProviders as $sp) {
            $sp->updateRating();
        }
    }

    private function createAdminUser(ObjectManager $manager): void
    {
        $admin = new User();
        $admin->setEmail('admin@homehelper.com');
        $admin->setFirstName('Admin');
        $admin->setLastName('User');
        $admin->setPhone('(555) 000-0000');
        $admin->setRoles(['ROLE_ADMIN', 'ROLE_SUPER_ADMIN']);
        $admin->setPassword(
            $this->passwordHasher->hashPassword($admin, 'admin123')
        );

        $manager->persist($admin);
    }
}
