<?php

namespace App\Form;

use App\Entity\Booking;
use App\Entity\Service;
use App\Entity\ServiceProvider;
use Symfony\Bridge\Doctrine\Form\Type\EntityType;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\Extension\Core\Type\DateType;
use Symfony\Component\Form\Extension\Core\Type\EmailType;
use Symfony\Component\Form\Extension\Core\Type\TelType;
use Symfony\Component\Form\Extension\Core\Type\TextareaType;
use Symfony\Component\Form\Extension\Core\Type\TextType;
use Symfony\Component\Form\Extension\Core\Type\TimeType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\OptionsResolver\OptionsResolver;

class AdminBookingType extends AbstractType
{
    public function buildForm(FormBuilderInterface $builder, array $options): void
    {
        $builder
            ->add('serviceProvider', EntityType::class, [
                'class' => ServiceProvider::class,
                'choice_label' => function(ServiceProvider $sp) {
                    return $sp->getUser()->getFullName() . ' ($' . $sp->getHourlyRate() . '/hr)';
                },
                'label' => 'Service Provider',
                'placeholder' => 'Select service provider...',
            ])
            ->add('service', EntityType::class, [
                'class' => Service::class,
                'choice_label' => function(Service $service) {
                    return $service->getIcon() . ' ' . $service->getName();
                },
                'label' => 'Service',
                'placeholder' => 'Select service...',
            ])
            ->add('customerName', TextType::class, [
                'label' => 'Customer Name',
                'attr' => ['placeholder' => 'Full name']
            ])
            ->add('customerEmail', EmailType::class, [
                'label' => 'Customer Email',
                'attr' => ['placeholder' => 'email@example.com']
            ])
            ->add('customerPhone', TelType::class, [
                'label' => 'Customer Phone',
                'attr' => ['placeholder' => '(555) 123-4567']
            ])
            ->add('serviceDate', DateType::class, [
                'label' => 'Service Date',
                'widget' => 'single_text',
                'html5' => true,
            ])
            ->add('serviceTime', TimeType::class, [
                'label' => 'Service Time',
                'widget' => 'single_text',
                'html5' => true,
            ])
            ->add('hours', ChoiceType::class, [
                'label' => 'Duration (Hours)',
                'choices' => [
                    '2 hours' => 2,
                    '3 hours' => 3,
                    '4 hours' => 4,
                    '5 hours' => 5,
                    '6 hours' => 6,
                    '8 hours' => 8,
                    '10 hours' => 10,
                ],
            ])
            ->add('serviceAddress', TextareaType::class, [
                'label' => 'Service Address',
                'attr' => [
                    'rows' => 3,
                    'placeholder' => 'Full address where service will be performed'
                ]
            ])
            ->add('specialInstructions', TextareaType::class, [
                'label' => 'Special Instructions',
                'required' => false,
                'attr' => [
                    'rows' => 3,
                    'placeholder' => 'Any special requirements...'
                ]
            ])
            ->add('status', ChoiceType::class, [
                'label' => 'Booking Status',
                'choices' => [
                    'Pending' => Booking::STATUS_PENDING,
                    'Confirmed' => Booking::STATUS_CONFIRMED,
                    'Completed' => Booking::STATUS_COMPLETED,
                    'Cancelled' => Booking::STATUS_CANCELLED,
                ],
            ])
        ;
    }

    public function configureOptions(OptionsResolver $resolver): void
    {
        $resolver->setDefaults([
            'data_class' => Booking::class,
        ]);
    }
}
